import { NextApiRequest, NextApiResponse } from "next";

import { requireSession } from "@/lib/auth";
import { z } from "zod";
import { prisma } from "@/lib/prisma";

const QuerySchema = z.object({
  elementId: z.string().cuid(),
  departmentId: z.string().cuid().optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireSession(req, res);
  if (!session) return;

  if (req.method !== "GET") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  const parsed = QuerySchema.safeParse(req.query);
  if (!parsed.success) {
    return res.status(400).json({
      ok: false,
      error: "Invalid input",
      details: parsed.error.flatten(),
    });
  }

  const { elementId, departmentId } = parsed.data;

  try {
    // Get element details
    const element = await prisma.element.findUnique({
      where: { id: elementId },
      include: {
        location: {
          include: {
            department: true,
          },
        },
      },
    });

    if (!element) {
      return res.status(404).json({ ok: false, error: "Element not found" });
    }

    // Find technicians whose roles are assigned to this element (case-insensitive role check)
    const allUsers = await prisma.user.findMany({
      where: {
        customRole: {
          isActive: true,
          roleElements: {
            some: {
              elementId: elementId,
            },
          },
        },
      },
      select: {
        id: true,
        name: true,
        email: true,
        hourlyRate: true,
        role: true,
        isExternal: true,
        companyName: true,
        customRole: {
          select: {
            id: true,
            name: true,
            description: true,
          },
        },
      },
      orderBy: [
        { hourlyRate: "asc" }, // Prefer lower cost technicians
        { name: "asc" },
      ],
    });

    // Filter to only TECHNICIAN role (case-insensitive)
    const technicians = allUsers.filter(
      (user) => user.role?.toUpperCase() === "TECHNICIAN"
    );

    // Also find technicians without specific roles but in the same department
    const allDeptUsers = departmentId
      ? await prisma.user.findMany({
          where: {
            customRoleId: null, // No specific role assigned
          },
          select: {
            id: true,
            name: true,
            email: true,
            hourlyRate: true,
            role: true,
            isExternal: true,
            companyName: true,
            // @ts-ignore
            customRole: null,
          },
          orderBy: [{ hourlyRate: "asc" }, { name: "asc" }],
        })
      : [];

    // Filter to only TECHNICIAN role (case-insensitive)
    const departmentTechnicians = allDeptUsers
      ? allDeptUsers.filter((user) => user.role?.toUpperCase() === "TECHNICIAN")
      : [];

    // Combine and deduplicate technicians (remove role field)
    const allTechnicians = [
      ...technicians.map((t) => {
        const { role, ...rest } = t;
        return { ...rest, matchType: "expert" as const };
      }),
      ...departmentTechnicians.map((t) => {
        const { role, ...rest } = t;
        return { ...rest, matchType: "general" as const };
      }),
    ];

    // Remove duplicates based on user ID
    const uniqueTechnicians = allTechnicians.filter(
      (tech, index, self) => index === self.findIndex((t) => t.id === tech.id)
    );

    return res.status(200).json({
      ok: true,
      element: {
        id: element.id,
        name: element.name,
        department: element.location?.department?.name || "—",
        location: element.location?.name || "—",
      },
      technicians: uniqueTechnicians,
      suggestions: {
        expert: technicians.length,
        general: departmentTechnicians.length,
        total: uniqueTechnicians.length,
      },
    });
  } catch (error) {
    console.error("Error suggesting technicians:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to suggest technicians",
    });
  }
}
